<?php
// 定时更新海报
header('Content-Type: application/json; charset=utf-8');

// 引入共享函数库
require_once __DIR__ . '/common_functions.php';

// 配置文件路径
$configDir = __DIR__ . '/../config';
$logsDir = __DIR__ . '/../logs';
$configFile = $configDir . '/schedule.json';
$logFile = $logsDir . '/schedule.log';

// 确保目录存在
function ensureDirectoryExists($dir) {
    if (!is_dir($dir)) {
        $created = @mkdir($dir, 0755, true);
        if (!$created) {
            return false;
        }
    }
    return is_writable($dir);
}

// 写入日志
function writeLog($message) {
    global $logFile, $logsDir;
    
    // 如果目录不存在或不可写，尝试使用临时目录
    if (!ensureDirectoryExists(dirname($logFile))) {
        $tempDir = sys_get_temp_dir();
        $logFile = $tempDir . '/haibao_schedule.log';
    }
    
    $date = date('Y-m-d H:i:s');
    $logMessage = "[$date] $message\n";
    @file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// 验证访问密钥
$correctKey = md5('haibao_schedule_key');
$inputKey = isset($_GET['key']) ? $_GET['key'] : '';

if ($inputKey !== $correctKey) {
    writeLog('访问密钥无效: ' . $inputKey);
    outputJson([], 1, '无效的访问密钥');
}

// 确保配置目录存在
if (!ensureDirectoryExists($configDir)) {
    writeLog("配置目录不可写: $configDir");
    outputJson([], 1, "配置目录不可写，请检查权限 $configDir");
}

// 检查配置是否存在
if (!file_exists($configFile)) {
    // 配置不存在时，自动创建默认配置
    $defaultConfig = [
        'enabled' => true,  // 默认启用
        'frequency' => 'daily',
        'typeId' => 0,
        'count' => 10,
        'recommendLevel' => 9,
        'clearBeforeUpdate' => false,
        'lastUpdate' => 0
    ];
    
    $result = @file_put_contents($configFile, json_encode($defaultConfig, JSON_PRETTY_PRINT));
    if ($result === false) {
        writeLog("无法创建配置文件: $configFile");
        outputJson([], 1, "无法创建配置文件，请检查权限");
    } else {
        writeLog("已创建默认配置文件");
    }
}

// 读取配置
$config = @json_decode(file_get_contents($configFile), true);
if (!is_array($config)) {
    writeLog("配置文件格式错误");
    outputJson([], 1, "配置文件格式错误，请检查JSON格式");
}

// 检查是否启用 - 确保正确处理布尔值
$isEnabled = isset($config['enabled']) ? $config['enabled'] : false;
if (is_string($isEnabled)) {
    // 有可能布尔值被存储为字符串 "true" 或 "false"
    $isEnabled = ($isEnabled === "true" || $isEnabled === "1");
}

if (!$isEnabled) {
    writeLog('定时更新未启用，配置值: ' . var_export($config['enabled'], true));
    outputJson([], 1, '定时更新未启用，请在配置中开启');
}

// 开始执行更新
writeLog('开始执行海报更新');

// 使用通用函数获取数据库配置
$dbConfig = getSystemDbConfig();

// 连接数据库
try {
    $conn = new mysqli($dbConfig['dbHost'], $dbConfig['dbUser'], $dbConfig['dbPass'], $dbConfig['dbName'], $dbConfig['dbPort']);
    
    if ($conn->connect_error) {
        writeLog('数据库连接失败: ' . $conn->connect_error);
        outputJson([], 1, '数据库连接失败: ' . $conn->connect_error);
    }
    
    // 设置字符集
    $conn->set_charset('utf8');
    
    // 如果配置了清空当前推荐，先执行清空
    if (isset($config['clearBeforeUpdate']) && $config['clearBeforeUpdate']) {
        writeLog('执行清空当前推荐');
        $clearSql = "UPDATE mac_vod SET vod_level = 0 WHERE vod_level > 0";
        $conn->query($clearSql);
    }
    
    // 从海报网获取数据
    $haibaoData = getHaibaoData();
    
    if (empty($haibaoData)) {
        writeLog('海报网返回的数据为空');
        outputJson([], 1, "海报网返回的数据为空");
    }
    
    // 如果指定了分类，获取分类名称
    $typeName = '';
    if (!empty($config['typeId'])) {
        $typeName = getTypeName($config['typeId'], $conn);
    }
    
    // 过滤符合条件的海报
    $filteredData = [];
    foreach ($haibaoData as $item) {
        if (empty($config['typeId']) || $item['source'] === $typeName) {
            $filteredData[] = $item;
        }
    }
    
    if (empty($filteredData)) {
        writeLog('没有找到符合条件的海报');
        outputJson([], 1, "没有找到符合条件的海报");
    }
    
    // 随机选择海报
    $updatedMovies = [];
    $maxAttempts = 100; // 设置最大尝试次数，防止无限循环
    
    shuffle($filteredData); // 随机排序数组
    
    $count = min(count($filteredData), isset($config['count']) ? intval($config['count']) : 10);
    $attempts = 0;
    
    while (count($updatedMovies) < $count && $attempts < $maxAttempts) {
        if (empty($filteredData)) {
            break; // 如果没有更多海报可选
        }
        
        $randomItem = array_shift($filteredData); // 取出第一个元素
        $movieName = $randomItem['name'];
        
        // 查询数据库中是否存在相同的电影名称
        $sql = "SELECT vod_id, vod_name FROM mac_vod WHERE vod_name = ?";
        if (!empty($config['typeId'])) {
            $sql .= " AND type_id = ?";
        }
        
        $stmt = $conn->prepare($sql);
        
        if ($stmt === false) {
            $attempts++;
            continue; // 跳过错误的条目
        }
        
        if (!empty($config['typeId'])) {
            $stmt->bind_param("si", $movieName, $config['typeId']);
        } else {
            $stmt->bind_param("s", $movieName);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $movie = $result->fetch_assoc();
            // 更新推荐等级和海报链接
            $updateSql = "UPDATE mac_vod SET vod_level = ?, vod_pic_slide = ? WHERE vod_id = ?";
            $updateStmt = $conn->prepare($updateSql);
            
            if ($updateStmt !== false) {
                $recommendLevel = isset($config['recommendLevel']) ? intval($config['recommendLevel']) : 9;
                $updateStmt->bind_param("isi", $recommendLevel, $randomItem['url'], $movie['vod_id']);
                $updateStmt->execute();
                $updatedMovies[] = [
                    'name' => $movie['vod_name'],
                    'url' => $randomItem['url']
                ];
                writeLog('成功更新影片: ' . $movie['vod_name']);
            }
        }
        
        $attempts++;
    }
    
    // 更新最后执行时间
    $config['lastUpdate'] = time();
    file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT));
    
    // 返回结果
    if (!empty($updatedMovies)) {
        $message = '成功更新 ' . count($updatedMovies) . ' 个海报';
        writeLog($message);
        outputJson($updatedMovies, 0, $message);
    } else {
        $message = '没有找到可更新的影片';
        writeLog($message);
        outputJson([], 1, $message);
    }
    
    $conn->close();
    
} catch (Exception $e) {
    $message = "执行更新时发生错误: " . $e->getMessage();
    writeLog($message);
    outputJson([], 1, $message);
} 